$ErrorActionPreference = 'Stop'

$resExistErr = [System.Management.Automation.ErrorCategory]::ResourceExists

# Set folder and file paths

$officetempfolder = Join-Path $env:Temp 'Office_temp'
$officetempfolderextract  = Join-Path $officetempfolder 'extract'
$configurationFile = Join-Path $officetempfolder 'config.xml'
$ModifiedconfigurationFile = Join-Path $officetempfolder 'modconfig.xml'
$ISOFile = Join-Path $officetempfolder 'office.iso'

Write-Verbose "tempenv: $env:Temp"
Write-Verbose "temp folder: $officetempfolder"
Write-Verbose "temp folderex: $officetempfolderextract "


# Get parameters

$pp = Get-PackageParameters
$WebconfigurationFile = $pp["configfile"]  #(Required) Configfile url, see Get-ChocolateyWebFile for url types.
$configurationFileChecksum = $pp["configchecksum"]   #(Required) Configfile checksum by sha256 algorithm.
$WebIsoFile = $pp["isofile"]  #(Required) ISO disc image url (can be any 7z supported archive), see Get-ChocolateyWebFile for url types.
$IsoChecksum = $pp["isochecksum"]  #(Required) ISO disc image checksum by sha256 algorithm.
$IgnoreCertIso = $pp["ignorecertiso"] #(Optional) <True/False> Don't check certificates when using HTTPS for the ISO disc image download. Will use different downloader code.
$IgnoreCertConf = $pp["ignorecertconf"] #(Optional) <True/False> Don't check certificates when using HTTPS for the configuration file download. Will use different downloader code.
$ActivateOffice = $pp["activate"] #(Optional) <True/False> Try activating office after install.
$OfficeKMSserver = $pp["kmsserver"] #(Optional) <hostname/IP> Set the definied KMS server.

#Validate parameters

if (-not $WebconfigurationFile)
{
	throw "This package requires a configfile."
}

if (-not $configurationFileChecksum)
{
	throw "This package requires a configfile checksum."
}

if (-not $WebIsoFile)
{
	throw "This package requires an install disc ISO."
}

if (-not $IsoChecksum)
{
	throw "This package requires an install disc ISO checksum."
}

if ($IgnoreCertIso)
{
	switch ($IgnoreCertIso)
	{
		'False' {$IgnoreCertIso = $false}
		'True'  {$IgnoreCertIso = $true}
		Default {$IgnoreCertIso = $false}
	}
}
else
{
	$IgnoreCertIso = $false
}

if ($IgnoreCertConf)
{
	switch ($IgnoreCertConf)
	{
		'False' {$IgnoreCertConf = $false}
		'True'  {$IgnoreCertConf = $true}
		Default {$IgnoreCertConf = $false}
	}
}
else
{
	$IgnoreCertConf = $false
}

if ($ActivateOffice)
{
	switch ($ActivateOffice)
	{
		'False' {$ActivateOffice = $false}
		'True'  {$ActivateOffice = $true}
		Default {$ActivateOffice = $false}
	}
}

if (-not $OfficeKMSserver)
{
	$OfficeKMSserver = $false
}

# Remove extracted files from previous (possibly failed) installation.

if (Test-Path "$officetempfolderextract") {
	Write-Verbose "Temp folder $officetempfolderextract already exists."
    Remove-Item -Recurse "$officetempfolderextract"
}

# Create work folder

if ((Test-Path "$officetempfolder") -eq $false)
{
	try
	{
		New-Item -ItemType directory -Path $officetempfolder
	}
	catch 
	{
		throw "Can't create temp folder $officetempfolder exiting."
	}
}

# Download the deployment files
#
# Get-ChocolateyWebFile requires valid certificates for https.


#Download the config file
if ($IgnoreCertConf -eq $False)
{
	#Get-ChocolateyWebFile automatically handles redownload and the integrity check on existing files.
	Get-ChocolateyWebFile `
	-PackageName "Microsoft.Office" `
	-FileFullPath $configurationFile `
	-Url64bit $WebconfigurationFile `
	-Checksum64 $configurationFileChecksum `
	-ChecksumType64 sha256
}
elseif ($IgnoreCertConf -eq $True)
{
	#Check if old file is intact (or even exists)
	$ErrorActionPreference = 'Continue'
	try
	{
		Get-ChecksumValid -File $configurationFile -CheckSum $configurationFileChecksum -ChecksumType sha256
	}
	catch
	{
		Write-Verbose "Checksums didn't match on $configurationFile redownloading."
		Remove-Item $configurationFile
	}
	finally
	{
		$ErrorActionPreference = 'Stop'
	}
	
	#Redownload if file doesn't exists
	if ((Test-Path "$configurationFile") -eq $false)
	{
		$tmp = [Net.ServicePointManager]::SecurityProtocol
		[Net.ServicePointManager]::SecurityProtocol = "tls13, tls12, tls11, tls"
		[System.Net.ServicePointManager]::ServerCertificateValidationCallback = { $true }
		$ProgressPreference = 'SilentlyContinue'
	
		try
		{
			Invoke-WebRequest -Uri $WebconfigurationFile -OutFile $configurationFile -ErrorAction Stop
		}
		catch
		{
			Write-Verbose "Error downloading $WebconfigurationFile to $configurationFile"
			Write-Verbose $_
		}
		finally
		{
			[Net.ServicePointManager]::SecurityProtocol = 0
			[System.Net.ServicePointManager]::ServerCertificateValidationCallback = $null
			[Net.ServicePointManager]::SecurityProtocol = $tmp
			$tmp = $null
		}
	
		Get-ChecksumValid -File $configurationFile -CheckSum $configurationFileChecksum -ChecksumType sha256		
	}
}


#Download the ISO file
if ($IgnoreCertIso -eq $False)
{
	#Get-ChocolateyWebFile automatically handles redownload and the integrity check on existing files.
	Get-ChocolateyWebFile `
	-PackageName "Microsoft.Office" `
	-FileFullPath $ISOFile `
	-Url64bit $WebIsoFile `
	-Checksum64 $IsoChecksum `
	-ChecksumType64 sha256
}
elseif ($IgnoreCertIso -eq $True)
{
	#Check if old file is intact (or even exists)
	$ErrorActionPreference = 'Continue'
	try
	{
		Get-ChecksumValid -File $ISOFile -CheckSum $IsoChecksum -ChecksumType sha256
	}
	catch
	{
		Write-Verbose "Checksums didn't match on $ISOFile redownloading."
		Remove-Item $ISOFile
	}
	finally
	{
		$ErrorActionPreference = 'Stop'
	}
	
	#Redownload if file doesn't exists
	if ((Test-Path "$ISOFile") -eq $false)
	{
		$tmp = [Net.ServicePointManager]::SecurityProtocol
		[Net.ServicePointManager]::SecurityProtocol = "tls13, tls12, tls11, tls"
		[System.Net.ServicePointManager]::ServerCertificateValidationCallback = { $true }
		$ProgressPreference = 'SilentlyContinue'
	
		try
		{
			Invoke-WebRequest -Uri $WebIsoFile -OutFile $ISOFile -ErrorAction Stop
		}
		catch
		{
			Write-Verbose "Error downloading $WebIsoFile to $ISOFile"
			Write-Verbose $_
		}
		finally
		{
			[Net.ServicePointManager]::SecurityProtocol = 0
			[System.Net.ServicePointManager]::ServerCertificateValidationCallback = $null
			[Net.ServicePointManager]::SecurityProtocol = $tmp
			$tmp = $null
		}
	
		Get-ChecksumValid -File $ISOFile -CheckSum $IsoChecksum -ChecksumType sha256		
	}
}


#Extract ISO file

Get-ChocolateyUnzip `
-FileFullPath64 $ISOFile `
-destination $officetempfolderextract


#Set SourcePath to $officetempfolderextract in the Configfile

$x = [xml] (Get-Content $configurationFile)
$node = $x.SelectNodes("/Configuration/Add")
$node.SetAttribute("SourcePath", $officetempfolderextract)
$x.Save($ModifiedconfigurationFile)


# Run the actual Office setup

$packageArgs                = @{
    packageName             = 'Microsoft.Office'
    fileType                = 'exe'
    file					= "$officetempfolderextract\Setup.exe"
    checksumType            = 'sha256'
    silentArgs				= "/configure $ModifiedconfigurationFile"
	UseOnlyPackageSilentArguments = $true
	validExitCodes          = @(
        0, # success
        3010, # success, restart required
        2147781575, # pending restart required
        2147205120  # pending restart required for setup update
    )
}

Install-ChocolateyInstallPackage @packageArgs


#Set KMS server if specified

if ($OfficeKMSserver -ne $false)
{
	Write-Verbose "Setting KMS server"
    start-process -FilePath "cmd.exe" -ArgumentList "/c", "cscript", "`"C:\Program Files\Microsoft Office\Office16\ospp.vbs`"", "/sethst:$$OfficeKMSserver" `
    -Wait -Passthru -NoNewWindow -WorkingDirectory "C:\Program Files\Microsoft Office\Office16\"
}


#Activate Office if requested

if ($ActivateOffice -eq $true)
{
	Write-Verbose "Activating Office"
    start-process -FilePath "cmd.exe" -ArgumentList "/c", "cscript", "`"C:\Program Files\Microsoft Office\Office16\ospp.vbs`"", "/ato" `
    -Wait -Passthru -NoNewWindow -WorkingDirectory "C:\Program Files\Microsoft Office\Office16\"
}

#Cleanup FIX ME: Remove-Item is async
Write-Verbose "Cleaning up"
if (Test-Path "$officetempfolder") {
	$v=0
	$i=$true
	$ErrorActionPreference = 'Continue'
	while(($v -le 10) -and ($i -eq $true))
	{
		try
		{
			$i=$false
			Remove-Item -Recurse "$officetempfolder" -Force
		}
		catch
		{
			$i=$true
			Write-Verbose $_
			Write-Verbose "Cleanup failed retrying"
		}
		
		$v++
		
		start-sleep -seconds 1
	}
}